<?php

namespace App\Services;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Str;

class PdfGenerator
{
    private $pdfFolderDirectory;
    private $generated_file_name;

    public function __construct()
    {
        $this->pdfFolderDirectory = 'pdf_templates';
    }

    /**
     * Generate a PDF using the given template and data.
     *
     * @param array $data
     * @param string $templateName
     * @param string|array $paperSize
     * @param string $action
     * @param string|null $file_path
     * @return \Illuminate\Http\Response
     * @throws \InvalidArgumentException
     */
    public function generatePdf(array|object $data, string $templateName, $paperSize = 'a4', string $action, string $file_path = null)
    {
        $templateName = str_replace('.blade.php', '', $templateName);
        $pdf = Pdf::loadView("{$this->pdfFolderDirectory}.{$templateName}", compact('data'));
        
        $paperSize = is_array($paperSize)
            ? $this->paperSize(...$paperSize)
            : $paperSize;

        $pdf->setPaper($paperSize, 'portrait');

        $file_name = $this->generated_file_name ?? $this->getNameOfDownloadedPdf($templateName);

        return match ($action) {
            'download' => $pdf->download($file_name),
            'stream' => $pdf->stream($file_name),
            'save' => (is_null($file_path)) 
                    ? $pdf->download($file_name) 
                    : $pdf->save($file_name, $file_path ),
            default => throw new \InvalidArgumentException("Invalid action: {$action}"),
        };
    }

    /**
     * Returns the paper size in points, given a width and height in any unit.
     *
     * @param  float  $width
     * @param  float  $height
     * @param  string  $scale  The unit of measurement for the width and height. Supported values are 'mm', 'cm', 'inch'.
     * @return array
     */
    private function paperSize(float $width, float $height, string $scale = 'inch'): array
    {
        $pointsPerUnit = $this->getPointsPerUnit($scale);

        $paperWidth = $width * $pointsPerUnit;
        $paperHeight = $height * $pointsPerUnit;

        return [0, 0, $paperWidth, $paperHeight];
    }

    /**
     * Set the folder where the PDF template is located.
     *
     * @param  string  $template_folder_directory  The folder where the PDF template is located.
     * @return $this
     */
    public function setTemplateFolderDirectory(string $template_folder_directory)
    {
        $this->pdfFolderDirectory = $template_folder_directory;

        return $this;
    }

    /**
     * Returns the number of points per unit for the given unit of measurement.
     * In DomPDF package every inch equals to 72 points in screen size.
     *
     * @param  string  $unit  The unit of measurement. Supported values are 'mm', 'cm', 'inch'.
     * @return float
     */
    private function getPointsPerUnit(string $unit): float
    {
        return match ($unit) {
            'mm' => 72 / 25.4,
            'cm' => 72 / 2.54,
            'inch' => 72,
            default => 72,
        };
    }

    /**
     * Return the name of the generated PDF file. The name is the given file name
     * with '.pdf' appended.
     *
     * @param string $fileName The name of the generated PDF file.
     * @return string The name of the generated PDF file.
     */
    public function getNameOfDownloadedPdf($fileName)
    {
        return "{$fileName}.pdf";
    }

    /**
     * Set the name of the generated PDF file.
     *
     * @param string $fileName
     * @return $this
     */
    public function setGeneratedFileName($fileName)
    {
        $this->generated_file_name = Str::endsWith($fileName, '.pdf') ? $fileName : "{$fileName}.pdf";

        return $this;
    }
}
